<?php
/**
 * 产品品牌简码
 * 用于在产品详情页显示品牌信息
 */

namespace Tooto_Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Product_Brand_Shortcode {

	/**
	 * 初始化简码
	 */
	public static function init(): void {
		add_shortcode( 'tooto_product_brand', [ __CLASS__, 'render' ] );
	}

	/**
	 * 渲染简码
	 *
	 * @param array  $atts    简码属性
	 * @param string $content 简码内容（不使用）
	 * @return string
	 */
	public static function render( $atts, $content = '' ): string {
		// 解析属性
		$atts = shortcode_atts(
			[
				'product_id' => 0,        // 产品ID，0表示使用当前产品
				'show_label' => 'no',     // 是否显示"品牌："标签
				'label_text' => '',       // 自定义标签文本，为空则使用默认
				'separator'  => ', ',     // 多个品牌之间的分隔符
				'class'      => '',       // 自定义CSS类
			],
			$atts,
			'tooto_product_brand'
		);

		// 获取产品ID
		$product_id = absint( $atts['product_id'] );
		
		if ( ! $product_id ) {
			// 尝试从全局获取
			global $product;
			if ( $product && is_a( $product, 'WC_Product' ) ) {
				$product_id = $product->get_id();
			} elseif ( is_product() ) {
				global $post;
				$product_id = $post->ID;
			}
		}

		if ( ! $product_id ) {
			return '';
		}

		// 获取品牌信息
		$brands = self::get_product_brands( $product_id );

		if ( empty( $brands ) ) {
			return '';
		}

		// 构建输出
		$output = '<div class="tooto-product-brand' . ( $atts['class'] ? ' ' . esc_attr( $atts['class'] ) : '' ) . '">';

		// 显示标签
		if ( 'yes' === $atts['show_label'] ) {
			$label_text = ! empty( $atts['label_text'] ) ? $atts['label_text'] : __( '品牌：', 'tooto-elementor' );
			$output .= '<span class="tooto-product-brand__label">' . esc_html( $label_text ) . '</span>';
		}

		// 显示品牌列表（仅展示，不链接）
		$brand_items = [];
		foreach ( $brands as $brand ) {
			$brand_name = esc_html( $brand->name );
			$brand_items[] = '<span class="tooto-product-brand__name">' . $brand_name . '</span>';
		}

		$output .= '<span class="tooto-product-brand__list">' . implode( esc_html( $atts['separator'] ), $brand_items ) . '</span>';
		$output .= '</div>';

		return $output;
	}

	/**
	 * 获取产品的品牌信息
	 *
	 * @param int $product_id 产品ID
	 * @return array 品牌术语数组
	 */
	private static function get_product_brands( int $product_id ): array {
		// 尝试多种可能的品牌 taxonomy
		$possible_taxonomies = [
			'product_brand',      // WooCommerce Brands / Blocksy
			'pwb-brand',          // Perfect WooCommerce Brands
			'product_brands',     // 其他可能的变体
			'pa_brand',          // 作为产品属性
		];

		foreach ( $possible_taxonomies as $taxonomy ) {
			if ( taxonomy_exists( $taxonomy ) ) {
				$terms = get_the_terms( $product_id, $taxonomy );
				if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
					return $terms;
				}
			}
		}

		// 如果 taxonomy 不存在，尝试从产品属性中获取
		$product = wc_get_product( $product_id );
		if ( $product ) {
			$attributes = $product->get_attributes();
			foreach ( $attributes as $attribute ) {
				if ( $attribute->get_name() === 'pa_brand' || $attribute->get_name() === 'brand' ) {
					$terms = wc_get_product_terms( $product_id, $attribute->get_name(), [ 'fields' => 'all' ] );
					if ( ! is_wp_error( $terms ) && ! empty( $terms ) ) {
						return $terms;
					}
				}
			}
		}

		return [];
	}
}
