<?php
/**
 * 产品库存状态简码
 * 用于在产品卡片和产品详情页显示库存状态（In Stock / Restocking）
 */

namespace Tooto_Elementor;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

class Product_Stock_Status_Shortcode {

	/**
	 * 初始化简码
	 */
	public static function init(): void {
		add_shortcode( 'tooto_product_stock_status', [ __CLASS__, 'render' ] );
	}

	/**
	 * 渲染简码
	 *
	 * @param array  $atts    简码属性
	 * @param string $content 简码内容（不使用）
	 * @return string
	 */
	public static function render( $atts, $content = '' ): string {
		// 解析属性
		$atts = shortcode_atts(
			[
				'product_id' => 0,        // 产品ID，0表示使用当前产品
				'class'      => '',       // 自定义CSS类
			],
			$atts,
			'tooto_product_stock_status'
		);

		// 获取产品ID
		$product_id = absint( $atts['product_id'] );
		
		if ( ! $product_id ) {
			// 尝试从全局获取
			global $product;
			if ( $product && is_a( $product, 'WC_Product' ) ) {
				$product_id = $product->get_id();
			} elseif ( is_product() ) {
				global $post;
				$product_id = $post->ID;
			}
		}

		if ( ! $product_id ) {
			return '';
		}

		// 获取产品对象
		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return '';
		}

		// 获取库存状态
		$stock_status = self::get_stock_status( $product );

		if ( empty( $stock_status ) ) {
			return '';
		}

		// 构建输出
		$status_class = 'tooto-product-stock-status--' . esc_attr( $stock_status['type'] );
		$custom_class = ! empty( $atts['class'] ) ? ' ' . esc_attr( $atts['class'] ) : '';
		
		$output = '<div class="tooto-product-stock-status ' . $status_class . $custom_class . '" data-stock-status="' . esc_attr( $stock_status['type'] ) . '">';
		
		// 图标
		$output .= '<span class="tooto-product-stock-status__icon">';
		$output .= self::get_status_icon( $stock_status['type'] );
		$output .= '</span>';
		
		// 文本
		$output .= '<span class="tooto-product-stock-status__text">' . esc_html( $stock_status['label'] ) . '</span>';
		
		$output .= '</div>';

		return $output;
	}

	/**
	 * 获取产品的库存状态
	 *
	 * @param \WC_Product $product 产品对象
	 * @return array 状态信息 ['type' => 'in-stock'|'restocking', 'label' => 'In Stock'|'Restocking']
	 */
	private static function get_stock_status( $product ): array {
		// 获取 WooCommerce 标准库存状态
		$stock_status = $product->get_stock_status();

		// 判断状态：使用产品管理的库存状态
		// 有货 (instock) → In Stock (#078100)
		// 缺货 (outofstock) → Out of Stock (#6F6F6F)
		// 预售 (onbackorder) → Restocking (#FF7300)
		
		if ( 'instock' === $stock_status ) {
			// 有库存
			return [
				'type'  => 'in-stock',
				'label' => __( 'In Stock', 'tooto-elementor' ),
			];
		} elseif ( 'outofstock' === $stock_status ) {
			// 缺货 → Out of Stock
			return [
				'type'  => 'out-of-stock',
				'label' => __( 'Out of Stock', 'tooto-elementor' ),
			];
		} elseif ( 'onbackorder' === $stock_status ) {
			// 延迟交货/预售 → Restocking
			return [
				'type'  => 'restocking',
				'label' => __( 'Restocking', 'tooto-elementor' ),
			];
		} elseif ( 'onpreorder' === $stock_status ) {
			// 预售 (Pre-Order)
			return [
				'type'  => 'pre-order',
				'label' => __( 'Pre-Order', 'tooto-elementor' ),
			];
		}

		// 默认返回空（不显示）
		return [];
	}

	/**
	 * 获取状态图标（内联 SVG，支持 currentColor）
	 *
	 * @param string $status_type 状态类型：'in-stock', 'restocking', 'out-of-stock' 或 'pre-order'
	 * @return string 图标 HTML
	 */
	private static function get_status_icon( string $status_type ): string {
		// 获取插件路径
		$plugin_path = defined( 'TOOTO_EA_PATH' ) ? TOOTO_EA_PATH : '';

		if ( 'in-stock' === $status_type ) {
			// In Stock 图标
			$svg_path = $plugin_path . 'assets/images/in-stock.svg';
		} elseif ( 'restocking' === $status_type ) {
			// Restocking 图标
			$svg_path = $plugin_path . 'assets/images/restocking.svg';
		} elseif ( 'pre-order' === $status_type ) {
			// Pre-Order 图标
			$svg_path = $plugin_path . 'assets/images/pre-sale.svg';
		} else {
			// Out of Stock 图标
			$svg_path = $plugin_path . 'assets/images/out-of-stock.svg';
			// 如果没有专用图标，回退到 restocking
			if ( ! file_exists( $svg_path ) ) {
				$svg_path = $plugin_path . 'assets/images/restocking.svg';
			}
		}

		// 读取 SVG 文件内容
		if ( ! empty( $plugin_path ) && file_exists( $svg_path ) ) {
			$svg_content = file_get_contents( $svg_path );
			if ( $svg_content ) {
				// 使用 preg_replace_callback 安全地修改 <svg> 根标签
				// 避免误删 clipPath 等内部元素中的 width/height 属性
				$svg_content = preg_replace_callback( '/<svg([^>]*)>/', function( $matches ) {
					$attributes = $matches[1];
					
					// 移除根标签中原有的 width, height, fill 属性
					$attributes = preg_replace( '/\s+width="[^"]*"/', '', $attributes );
					$attributes = preg_replace( '/\s+height="[^"]*"/', '', $attributes );
					$attributes = preg_replace( '/\s+fill="[^"]*"/', '', $attributes );
					
					// 返回带有强制属性的新标签
					// width/height: 24px
					// fill: currentColor (跟随文本颜色)
					return '<svg' . $attributes . ' width="24" height="24" fill="currentColor">';
				}, $svg_content, 1 );
				
				return $svg_content;
			}
		}

		// 如果文件读取失败，返回空字符串
		return '';
	}
}
