<?php
namespace Tooto_Elementor\Widgets;

use Elementor\Widget_Base;
use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Box_Shadow;
use Elementor\Group_Control_Image_Size;
use Elementor\Repeater;
use Elementor\Utils;

if ( ! defined( 'ABSPATH' ) ) {
	exit; // Exit if accessed directly.
}

class Hover_Image_Menu extends Widget_Base {

	public function get_name() {
		return 'tooto-hover-image-menu';
	}

	public function get_title() {
		return esc_html__( 'Hover Image Menu', 'tooto-elementor' );
	}

	public function get_icon() {
		return 'eicon-image-box';
	}

	public function get_categories() {
		return [ 'tooto' ];
	}

	public function get_keywords() {
		return [ 'hover', 'image', 'menu', 'link', 'tooto' ];
	}

	protected function register_controls() {
		$this->start_controls_section(
			'section_content',
			[
				'label' => esc_html__( 'Content', 'tooto-elementor' ),
				'tab' => Controls_Manager::TAB_CONTENT,
			]
		);

		$repeater = new Repeater();

		$repeater->add_control(
			'text',
			[
				'label' => esc_html__( 'Text', 'tooto-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => esc_html__( 'List Item', 'tooto-elementor' ),
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'link',
			[
				'label' => esc_html__( 'Link', 'tooto-elementor' ),
				'type' => Controls_Manager::URL,
				'dynamic' => [
					'active' => true,
				],
				'placeholder' => esc_html__( 'https://your-link.com', 'tooto-elementor' ),
				'default' => [
					'url' => '#',
				],
			]
		);

		$repeater->add_control(
			'image',
			[
				'label' => esc_html__( 'Image', 'tooto-elementor' ),
				'type' => Controls_Manager::MEDIA,
				'default' => [
					'url' => Utils::get_placeholder_image_src(),
				],
			]
		);

		$this->add_control(
			'items',
			[
				'label' => esc_html__( 'Menu Items', 'tooto-elementor' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'text' => esc_html__( 'Item #1', 'tooto-elementor' ),
						'image' => [
							'url' => Utils::get_placeholder_image_src(),
						],
					],
					[
						'text' => esc_html__( 'Item #2', 'tooto-elementor' ),
						'image' => [
							'url' => Utils::get_placeholder_image_src(),
						],
					],
				],
				'title_field' => '{{{ text }}}',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_layout',
			[
				'label' => esc_html__( 'Layout', 'tooto-elementor' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'container_height',
			[
				'label' => esc_html__( 'Min Height', 'tooto-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'vh', 'vw' ],
				'range' => [
					'px' => [
						'min' => 100,
						'max' => 1000,
					],
					'vh' => [
						'min' => 10,
						'max' => 100,
					],
					'vw' => [
						'min' => 10,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 400,
				],
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu' => 'min-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'list_width',
			[
				'label' => esc_html__( 'List Width (%)', 'tooto-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ '%' ],
				'range' => [
					'%' => [
						'min' => 10,
						'max' => 90,
					],
				],
				'default' => [
					'unit' => '%',
					'size' => 40,
				],
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu__list' => 'flex: 0 0 {{SIZE}}%; max-width: {{SIZE}}%;',
					'{{WRAPPER}} .tooto-hover-image-menu__images' => 'flex: 0 0 calc(100% - {{SIZE}}%); max-width: calc(100% - {{SIZE}}%);',
				],
			]
		);

		$this->add_responsive_control(
			'gap',
			[
				'label' => esc_html__( 'Gap', 'tooto-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'em' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'default' => [
					'unit' => 'px',
					'size' => 20,
				],
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu' => 'display: flex; gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_list_items',
			[
				'label' => esc_html__( 'List Items', 'tooto-elementor' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'item_typography',
				'selector' => '{{WRAPPER}} .tooto-hover-image-menu__item',
			]
		);

		$this->start_controls_tabs( 'tabs_item_style' );

		$this->start_controls_tab(
			'tab_item_normal',
			[
				'label' => esc_html__( 'Normal', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'item_color',
			[
				'label' => esc_html__( 'Color', 'tooto-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu__item' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tab_item_hover',
			[
				'label' => esc_html__( 'Hover / Active', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'item_color_hover',
			[
				'label' => esc_html__( 'Color', 'tooto-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu__item:hover' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tooto-hover-image-menu__item.active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->end_controls_tabs();

		$this->add_responsive_control(
			'item_spacing',
			[
				'label' => esc_html__( 'Spacing Between Items', 'tooto-elementor' ),
				'type' => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range' => [
					'px' => [
						'min' => 0,
						'max' => 100,
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu__list .list__text' => 'display: flex; flex-direction: column; gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_image',
			[
				'label' => esc_html__( 'Image', 'tooto-elementor' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'image_object_fit',
			[
				'label' => esc_html__( 'Object Fit', 'tooto-elementor' ),
				'type' => Controls_Manager::SELECT,
				'options' => [
					'cover' => esc_html__( 'Cover', 'tooto-elementor' ),
					'contain' => esc_html__( 'Contain', 'tooto-elementor' ),
					'fill' => esc_html__( 'Fill', 'tooto-elementor' ),
				],
				'default' => 'contain',
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu__image' => 'object-fit: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Border::get_type(),
			[
				'name' => 'image_border',
				'selector' => '{{WRAPPER}} .tooto-hover-image-menu__images img',
			]
		);

		$this->add_control(
			'image_border_radius',
			[
				'label' => esc_html__( 'Border Radius', 'tooto-elementor' ),
				'type' => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors' => [
					'{{WRAPPER}} .tooto-hover-image-menu__images img' => 'border-radius: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Box_Shadow::get_type(),
			[
				'name' => 'image_box_shadow',
				'selector' => '{{WRAPPER}} .tooto-hover-image-menu__images img',
			]
		);

		$this->end_controls_section();
	}

	protected function render() {
		$settings = $this->get_settings_for_display();
		$items = $settings['items'];

		if ( empty( $items ) ) {
			return;
		}
		?>
		<div class="tooto-hover-image-menu tooto-animate-entrance">
			<div class="tooto-hover-image-menu__list">
				<div class="list__text">
					<?php foreach ( $items as $index => $item ) : 
						$link_url = $item['link']['url'];
						$target = $item['link']['is_external'] ? ' target="_blank"' : '';
						$nofollow = $item['link']['nofollow'] ? ' rel="nofollow"' : '';
						$active_class = $index === 0 ? ' active' : '';
						?>
						<a href="<?php echo esc_url( $link_url ); ?>" 
						class="tooto-hover-image-menu__item<?php echo esc_attr( $active_class ); ?>" 
						data-index="<?php echo esc_attr( $index ); ?>"
						<?php echo $target . $nofollow; ?>>
							<?php echo esc_html( $item['text'] ); ?>
						</a>
					<?php endforeach; ?>
				</div>
			</div>
			<div class="tooto-hover-image-menu__images">
				<?php foreach ( $items as $index => $item ) : 
					$image_url = $item['image']['url'];
					$active_class = $index === 0 ? ' active' : '';
					?>
					<img src="<?php echo esc_url( $image_url ); ?>" 
						 alt="<?php echo esc_attr( $item['text'] ); ?>" 
						 class="tooto-hover-image-menu__image<?php echo esc_attr( $active_class ); ?>"
						 data-index="<?php echo esc_attr( $index ); ?>">
				<?php endforeach; ?>
			</div>
		</div>
		<?php
	}
}
