<?php
/**
 * Tooto Loop Grid：复用 Elementor(Pro/Pro-Elements) Loop Grid 的布局/查询/分页控件与渲染，
 * 仅在移动端改为 scroll-snap 横向滚动，并增加自定义分页功能。
 */

namespace Tooto_Elementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Border;
use Elementor\Group_Control_Background;
use Elementor\Group_Control_Box_Shadow;

use Elementor\Icons_Manager;

if ( class_exists( '\ElementorPro\Modules\LoopBuilder\Widgets\Loop_Grid' ) ) {
	/**
	 * 优先：直接继承 Loop Grid（保持布局/查询/分页 1:1）。
	 */
	class Loop_Grid extends \ElementorPro\Modules\LoopBuilder\Widgets\Loop_Grid {
		public function get_name(): string {
			return 'tooto-loop-grid';
		}

		public function get_title(): string {
			return __( 'Tooto Loop Grid', 'tooto-elementor' );
		}

		public function get_categories(): array {
			return [ 'tooto' ];
		}

		public function get_icon(): string {
			return 'eicon-loop-builder';
		}

		public function get_style_depends(): array {
			$deps = parent::get_style_depends();
			$deps[] = 'tooto-elementor';
			return array_values( array_unique( $deps ) );
		}

		protected function register_controls() {
			parent::register_controls();

			// 仅新增一个移动端 scroll-snap 卡片宽度配置，其他布局/查询/分页保持与 Loop Grid 一致。
			$this->start_injection(
				[
					'of' => 'columns',
					'at' => 'after',
				]
			);

			$this->add_responsive_control(
				'tooto_mobile_snap_item_width',
				[
					'label'      => esc_html__( '移动端卡片宽度', 'tooto-elementor' ),
					'type'       => Controls_Manager::SLIDER,
					'size_units' => [ '%', 'px' ],
					'range'      => [
						'%' => [
							'min' => 30,
							'max' => 100,
						],
						'px' => [
							'min' => 160,
							'max' => 520,
						],
					],
					'default'    => [
						'size' => 82,
						'unit' => '%',
					],
					'selectors'  => [
						'{{WRAPPER}}' => '--tooto-snap-item-width: {{SIZE}}{{UNIT}};',
					],
					'frontend_available' => true,
					'condition' => [
						'tooto_loop_carousel_enabled!' => 'yes',
					],
				]
			);

			$this->end_injection();

			// Carousel Controls
			$this->start_controls_section(
				'section_tooto_carousel',
				[
					'label' => esc_html__( 'Carousel Settings', 'tooto-elementor' ),
					'tab' => Controls_Manager::TAB_CONTENT,
				]
			);

			$this->add_control(
				'tooto_loop_carousel_enabled',
				[
					'label' => esc_html__( 'Enable Carousel', 'tooto-elementor' ),
					'type' => Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Yes', 'tooto-elementor' ),
					'label_off' => esc_html__( 'No', 'tooto-elementor' ),
					'return_value' => 'yes',
					'default' => '',
					'frontend_available' => true,
				]
			);

			$this->add_responsive_control(
				'tooto_carousel_slides_per_view',
				[
					'label' => esc_html__( 'Slides Per View', 'tooto-elementor' ),
					'type' => Controls_Manager::SELECT,
					'options' => [
						'1' => '1',
						'2' => '2',
						'3' => '3',
						'4' => '4',
						'5' => '5',
						'6' => '6',
					],
					'default' => '4',
					'tablet_default' => '2',
					'mobile_default' => '1',
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
					'frontend_available' => true,
				]
			);

			$this->add_responsive_control(
				'tooto_carousel_gap',
				[
					'label' => esc_html__( 'Gap (px)', 'tooto-elementor' ),
					'type' => Controls_Manager::SLIDER,
					'range' => [
						'px' => [
							'min' => 0,
							'max' => 100,
						],
					],
					'default' => [
						'size' => 20,
					],
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
					'frontend_available' => true,
				]
			);

			$this->add_control(
				'tooto_carousel_nav_position',
				[
					'label' => esc_html__( 'Navigation Position', 'tooto-elementor' ),
					'type' => Controls_Manager::SELECT,
					'options' => [
						'default' => esc_html__( 'Default (Sides)', 'tooto-elementor' ),
						'top-right' => esc_html__( 'Top Right (Header)', 'tooto-elementor' ),
					],
					'default' => 'default',
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
					'frontend_available' => true,
				]
			);

			$this->add_control(
				'tooto_carousel_infinite_loop',
				[
					'label' => esc_html__( 'Infinite Loop', 'tooto-elementor' ),
					'type' => Controls_Manager::SWITCHER,
					'return_value' => 'yes',
					'default' => '',
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
					'frontend_available' => true,
				]
			);

			$this->add_control(
				'tooto_carousel_mobile_behavior',
				[
					'label' => esc_html__( 'Mobile Behavior', 'tooto-elementor' ),
					'type' => Controls_Manager::SELECT,
					'options' => [
						'carousel' => esc_html__( 'Carousel', 'tooto-elementor' ),
						'load_more' => esc_html__( 'Grid + Load More', 'tooto-elementor' ),
					],
					'default' => 'carousel',
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
					'frontend_available' => true,
				]
			);

			$this->add_control(
				'tooto_mobile_load_more_text',
				[
					'label' => esc_html__( 'Load More Text', 'tooto-elementor' ),
					'type' => Controls_Manager::TEXT,
					'default' => esc_html__( 'Load More', 'tooto-elementor' ),
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
						'tooto_carousel_mobile_behavior' => 'load_more',
					],
					'frontend_available' => true,
				]
			);

			$this->add_control(
				'tooto_carousel_autoplay',
				[
					'label' => esc_html__( 'Autoplay', 'tooto-elementor' ),
					'type' => Controls_Manager::SWITCHER,
					'return_value' => 'yes',
					'default' => '',
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
					'frontend_available' => true,
				]
			);

			$this->add_control(
				'tooto_carousel_autoplay_speed',
				[
					'label' => esc_html__( 'Autoplay Speed (ms)', 'tooto-elementor' ),
					'type' => Controls_Manager::NUMBER,
					'default' => 3000,
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
						'tooto_carousel_autoplay' => 'yes',
					],
					'frontend_available' => true,
				]
			);
			
			$this->add_control(
				'tooto_carousel_prev_icon',
				[
					'label' => esc_html__( 'Previous Icon', 'tooto-elementor' ),
					'type' => Controls_Manager::ICONS,
					'default' => [
						'value' => 'fas fa-chevron-left',
						'library' => 'fa-solid',
					],
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
				]
			);

			$this->add_control(
				'tooto_carousel_next_icon',
				[
					'label' => esc_html__( 'Next Icon', 'tooto-elementor' ),
					'type' => Controls_Manager::ICONS,
					'default' => [
						'value' => 'fas fa-chevron-right',
						'library' => 'fa-solid',
					],
					'condition' => [
						'tooto_loop_carousel_enabled' => 'yes',
					],
				]
			);

			$this->end_controls_section();

			// Custom Pagination Controls
			$this->start_controls_section(
				'section_tooto_pagination',
				[
					'label' => esc_html__( 'Tooto Custom Pagination', 'tooto-elementor' ),
					'tab' => Controls_Manager::TAB_CONTENT,
				]
			);

			$this->add_control(
				'tooto_enable_custom_pagination',
				[
					'label' => esc_html__( 'Enable Custom Pagination', 'tooto-elementor' ),
					'type' => Controls_Manager::SWITCHER,
					'label_on' => esc_html__( 'Yes', 'tooto-elementor' ),
					'label_off' => esc_html__( 'No', 'tooto-elementor' ),
					'return_value' => 'yes',
					'default' => '',
					'description' => esc_html__( 'Please set "Pagination" to "None" in the "Pagination" section to avoid duplication.', 'tooto-elementor' ),
				]
			);

			$this->add_control(
				'tooto_pagination_prev_icon',
				[
					'label' => esc_html__( 'Previous Icon', 'tooto-elementor' ),
					'type' => Controls_Manager::ICONS,
					'default' => [
						'value' => 'fas fa-angle-left',
						'library' => 'fa-solid',
					],
					'condition' => [
						'tooto_enable_custom_pagination' => 'yes',
					],
				]
			);

			$this->add_control(
				'tooto_pagination_next_icon',
				[
					'label' => esc_html__( 'Next Icon', 'tooto-elementor' ),
					'type' => Controls_Manager::ICONS,
					'default' => [
						'value' => 'fas fa-angle-right',
						'library' => 'fa-solid',
					],
					'condition' => [
						'tooto_enable_custom_pagination' => 'yes',
					],
				]
			);

			$this->end_controls_section();

			// Style Tab
			$this->start_controls_section(
				'section_tooto_pagination_style',
				[
					'label' => esc_html__( 'Tooto Pagination', 'tooto-elementor' ),
					'tab' => Controls_Manager::TAB_STYLE,
					'condition' => [
						'tooto_enable_custom_pagination' => 'yes',
					],
				]
			);

			$this->add_group_control(
				Group_Control_Typography::get_type(),
				[
					'name' => 'tooto_pagination_typography',
					'selector' => '{{WRAPPER}} .tooto-pagination .page-numbers',
				]
			);

			$this->start_controls_tabs( 'tabs_tooto_pagination_style' );

			$this->start_controls_tab(
				'tab_tooto_pagination_normal',
				[
					'label' => esc_html__( 'Normal', 'tooto-elementor' ),
				]
			);

			$this->add_control(
				'tooto_pagination_color',
				[
					'label' => esc_html__( 'Text Color', 'tooto-elementor' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers:not(.current)' => 'color: {{VALUE}}',
						'{{WRAPPER}} .tooto-pagination-text' => 'color: {{VALUE}}',
						'{{WRAPPER}} .tooto-pagination-select' => 'color: {{VALUE}}; border-color: {{VALUE}};',
					],
				]
			);

			$this->add_control(
				'tooto_pagination_bg_color',
				[
					'label' => esc_html__( 'Background Color', 'tooto-elementor' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers:not(.current)' => 'background-color: {{VALUE}}',
					],
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'tab_tooto_pagination_hover',
				[
					'label' => esc_html__( 'Hover', 'tooto-elementor' ),
				]
			);

			$this->add_control(
				'tooto_pagination_hover_color',
				[
					'label' => esc_html__( 'Text Color', 'tooto-elementor' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers:hover' => 'color: {{VALUE}}',
					],
				]
			);

			$this->add_control(
				'tooto_pagination_hover_bg_color',
				[
					'label' => esc_html__( 'Background Color', 'tooto-elementor' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers:hover' => 'background-color: {{VALUE}}',
					],
				]
			);

			$this->end_controls_tab();

			$this->start_controls_tab(
				'tab_tooto_pagination_active',
				[
					'label' => esc_html__( 'Active', 'tooto-elementor' ),
				]
			);

			$this->add_control(
				'tooto_pagination_active_color',
				[
					'label' => esc_html__( 'Text Color', 'tooto-elementor' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers.current' => 'color: {{VALUE}}',
					],
				]
			);

			$this->add_control(
				'tooto_pagination_active_bg_color',
				[
					'label' => esc_html__( 'Background Color', 'tooto-elementor' ),
					'type' => Controls_Manager::COLOR,
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers.current' => 'background-color: {{VALUE}}',
					],
				]
			);

			$this->end_controls_tab();

			$this->end_controls_tabs();

			$this->add_responsive_control(
				'tooto_pagination_spacing',
				[
					'label' => esc_html__( 'Space Between', 'tooto-elementor' ),
					'type' => Controls_Manager::SLIDER,
					'range' => [
						'px' => [
							'max' => 50,
						],
					],
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers' => 'margin-right: {{SIZE}}{{UNIT}}; margin-left: {{SIZE}}{{UNIT}};',
						'{{WRAPPER}} .tooto-pagination' => 'gap: {{SIZE}}{{UNIT}};',
					],
				]
			);
			
			$this->add_control(
				'tooto_pagination_border_radius',
				[
					'label' => esc_html__( 'Border Radius', 'tooto-elementor' ),
					'type' => Controls_Manager::SLIDER,
					'size_units' => [ 'px', '%' ],
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers' => 'border-radius: {{SIZE}}{{UNIT}};',
					],
				]
			);

			$this->add_responsive_control(
				'tooto_pagination_padding',
				[
					'label' => esc_html__( 'Padding', 'tooto-elementor' ),
					'type' => Controls_Manager::DIMENSIONS,
					'size_units' => [ 'px', 'em', '%' ],
					'selectors' => [
						'{{WRAPPER}} .tooto-pagination .page-numbers' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
					],
				]
			);

			$this->end_controls_section();
		}

		/**
		 * 复用 Loop Grid 样式选择器（很多 CSS 以 elementor-widget-loop-grid 为锚点）。
		 */
		protected function get_html_wrapper_class() {
			$class = parent::get_html_wrapper_class();
			return $class . ' elementor-widget-loop-grid';
		}

		public $captured_query = null;

		public function capture_query_on_loop_start( $query ) {
			// We only want to capture the first valid query that runs during our render
			if ( null === $this->captured_query ) {
				$this->captured_query = $query;
			}
		}

		public function capture_query_on_the_post( $post, $query = null ) {
			// Capture query from the_post action as a fallback or alternative
			if ( null === $this->captured_query && $query instanceof \WP_Query ) {
				$this->captured_query = $query;
			}
		}

		/**
		 * 覆盖 render_content 方法
		 * 因为 Loop Grid 使用了 Skin 机制，直接覆盖 render() 会被跳过。
		 * 必须在 render_content() 中拦截。
		 */
		public function render_content() {
			$settings = $this->get_settings_for_display();

			// Hook into loop_start and the_post to capture the query object
			// 必须在父类渲染之前挂载，因为 Skin->render() 会在 parent::render_content() 中执行
			add_action( 'loop_start', [ $this, 'capture_query_on_loop_start' ] );
			add_action( 'the_post', [ $this, 'capture_query_on_the_post' ], 10, 2 );

			// Buffer output to inject pagination inside the container
			ob_start();
			parent::render_content();
			$content = ob_get_clean();

			// 清理 Hooks
			remove_action( 'loop_start', [ $this, 'capture_query_on_loop_start' ] );
			remove_action( 'the_post', [ $this, 'capture_query_on_the_post' ] );

			// Carousel Mode
			if ( ! empty( $settings['tooto_loop_carousel_enabled'] ) && 'yes' === $settings['tooto_loop_carousel_enabled'] ) {
				$this->render_carousel_mode( $content, $settings );
				return;
			}

			// 渲染自定义分页（注入到 .elementor-loop-container 内部）
			if ( ! empty( $settings['tooto_enable_custom_pagination'] ) && 'yes' === $settings['tooto_enable_custom_pagination'] ) {
				ob_start();
				$this->render_custom_pagination();
				$pagination_html = ob_get_clean();

				if ( ! empty( $pagination_html ) ) {
					// 尝试找到标准分页的位置，如果有，则 Loop Container 在它之前结束
					$split_marker = 'class="elementor-pagination"';
					$pos = strpos( $content, $split_marker );

					if ( $pos !== false ) {
						// 存在标准分页，Loop Container 在此之前结束
						// 在 $pos 之前的最后一个 </div> 前插入
						$before_pagination = substr( $content, 0, $pos );
						$last_div_pos = strrpos( $before_pagination, '</div>' );
						
						if ( $last_div_pos !== false ) {
							$content = substr_replace( $content, $pagination_html, $last_div_pos, 0 );
						} else {
							// 没找到结束标签（不太可能），直接追加到标准分页前
							$content = substr_replace( $content, $pagination_html, $pos, 0 ); // 这可能不太对，因为会插在 pagination 标签外
						}
					} else {
						// 没有标准分页，假设 Loop Container 是最外层的主要容器
						// 在最后一个 </div> 前插入
						$last_div_pos = strrpos( $content, '</div>' );
						if ( $last_div_pos !== false ) {
							$content = substr_replace( $content, $pagination_html, $last_div_pos, 0 );
						} else {
							$content .= $pagination_html;
						}
					}
				}
			}

			echo $content;
		}

		protected function render_carousel_mode( $content, $settings ) {
			if ( empty( $content ) ) {
				return;
			}

			$is_infinite = ! empty( $settings['tooto_carousel_infinite_loop'] ) && 'yes' === $settings['tooto_carousel_infinite_loop'];

			// Get Responsive Gaps
			$gap = isset( $settings['tooto_carousel_gap']['size'] ) ? (int) $settings['tooto_carousel_gap']['size'] . 'px' : '20px';
			$gap_tablet = isset( $settings['tooto_carousel_gap_tablet']['size'] ) ? (int) $settings['tooto_carousel_gap_tablet']['size'] . 'px' : $gap;
			$gap_mobile = isset( $settings['tooto_carousel_gap_mobile']['size'] ) ? (int) $settings['tooto_carousel_gap_mobile']['size'] . 'px' : $gap_tablet;

			$splide_options = [
				'type' => $is_infinite ? 'loop' : 'slide',
				'rewind' => false,
				'perPage' => isset( $settings['tooto_carousel_slides_per_view'] ) ? (int) $settings['tooto_carousel_slides_per_view'] : 4,
				'gap' => $gap,
				'pagination' => false,
				'arrows' => true,
				'autoplay' => false, // Explicitly disable autoplay by default
				'autoScroll' => false, // Explicitly disable autoScroll extension by default
				'breakpoints' => [
					1024 => [
						'perPage' => isset( $settings['tooto_carousel_slides_per_view_tablet'] ) ? (int) $settings['tooto_carousel_slides_per_view_tablet'] : 2,
						'gap' => $gap_tablet,
					],
					767 => [
						'perPage' => isset( $settings['tooto_carousel_slides_per_view_mobile'] ) ? (int) $settings['tooto_carousel_slides_per_view_mobile'] : 1,
						'gap' => $gap_mobile,
					],
				],
			];

			if ( ! empty( $settings['tooto_carousel_autoplay'] ) && 'yes' === $settings['tooto_carousel_autoplay'] ) {
				$splide_options['autoplay'] = true;
				$splide_options['interval'] = isset( $settings['tooto_carousel_autoplay_speed'] ) ? (int) $settings['tooto_carousel_autoplay_speed'] : 3000;
			}

			// Use DOMDocument to manipulate HTML structure
			$dom = new \DOMDocument();
			libxml_use_internal_errors( true );
			// Prefix with XML decl to force UTF-8
			$dom->loadHTML( '<?xml encoding="utf-8" ?>' . $content, LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
			libxml_clear_errors();

			$xpath = new \DOMXPath( $dom );
			$nodes = $xpath->query( '//*[contains(@class, "elementor-loop-container")]' );

			if ( $nodes->length > 0 ) {
				$container = $nodes->item( 0 );
				
				// Modify Container Classes
				$classes = $container->getAttribute( 'class' );
				$classes = str_replace( 'elementor-grid', '', $classes ); // Remove grid layout
				$classes .= ' splide__list';
				$container->setAttribute( 'class', $classes );

				// Add splide__slide to children (exclude style/script tags)
				foreach ( $container->childNodes as $child ) {
					if ( $child instanceof \DOMElement ) {
						// Skip non-visual elements like style, script, etc.
						if ( in_array( strtolower( $child->tagName ), [ 'style', 'script', 'link', 'meta' ], true ) ) {
							continue;
						}
						
						$childClass = $child->getAttribute( 'class' );
						$child->setAttribute( 'class', $childClass . ' splide__slide' );
					}
				}
				
				// Create Wrapper Structure
				$wrapper = $dom->createElement( 'div' );
				$nav_pos = isset( $settings['tooto_carousel_nav_position'] ) ? $settings['tooto_carousel_nav_position'] : 'default';
				$uid = 'tooto-loop-carousel-' . $this->get_id();
				$wrapper->setAttribute( 'class', 'tooto-loop-carousel splide splide-wrapper tooto-nav-' . $nav_pos . ' ' . $uid );
				$wrapper->setAttribute( 'data-splide', wp_json_encode( $splide_options ) );

				// Inject Style Block for Gap CSS Variable
				$style_content = ".{$uid} { --tooto-carousel-gap: {$gap}; } @media (max-width: 1024px) { .{$uid} { --tooto-carousel-gap: {$gap_tablet}; } } @media (max-width: 767px) { .{$uid} { --tooto-carousel-gap: {$gap_mobile}; } }";
				$style_el = $dom->createElement( 'style', $style_content );
				$wrapper->appendChild( $style_el );

				// Add Mobile Behavior data attribute
				if ( ! empty( $settings['tooto_carousel_mobile_behavior'] ) ) {
					$wrapper->setAttribute( 'data-mobile-behavior', $settings['tooto_carousel_mobile_behavior'] );
				}

				$track = $dom->createElement( 'div' );
				$track->setAttribute( 'class', 'splide__track' );

				// Move container into track
				$container->parentNode->replaceChild( $wrapper, $container );
				$wrapper->appendChild( $track );
				$track->appendChild( $container );

				// Add Arrows
				ob_start();
				Icons_Manager::render_icon( $settings['tooto_carousel_prev_icon'], [ 'aria-hidden' => 'true' ] );
				$prev_icon_html = ob_get_clean();
				if ( empty( $prev_icon_html ) ) $prev_icon_html = '<i class="fas fa-chevron-left"></i>';

				ob_start();
				Icons_Manager::render_icon( $settings['tooto_carousel_next_icon'], [ 'aria-hidden' => 'true' ] );
				$next_icon_html = ob_get_clean();
				if ( empty( $next_icon_html ) ) $next_icon_html = '<i class="fas fa-chevron-right"></i>';

				$arrows_html = sprintf(
					'<div class="splide__arrows"><button class="splide__arrow splide__arrow--prev">%s</button><button class="splide__arrow splide__arrow--next">%s</button></div>',
					$prev_icon_html,
					$next_icon_html
				);

				$arrows_frag = $dom->createDocumentFragment();
				// Use a temporary DOM to parse the arrows HTML
				$tempDom = new \DOMDocument();
				libxml_use_internal_errors( true );
				$tempDom->loadHTML( '<?xml encoding="utf-8" ?><body>' . $arrows_html . '</body>', LIBXML_HTML_NOIMPLIED | LIBXML_HTML_NODEFDTD );
				foreach ( $tempDom->getElementsByTagName( 'body' )->item( 0 )->childNodes as $node ) {
					 $importedNode = $dom->importNode( $node, true );
					 $arrows_frag->appendChild( $importedNode );
				}
				
				$wrapper->appendChild( $arrows_frag );
			}

			// Output modified HTML
			if ( $nodes->length > 0 ) {
				// Save only the wrapper if it exists (but we selected inner container?)
				// Actually we want to output the whole modified content.
				// Since we loaded the whole $content, saveHTML() returns it all.
				echo $dom->saveHTML();
				
				if ( ! empty( $settings['tooto_carousel_mobile_behavior'] ) && 'load_more' === $settings['tooto_carousel_mobile_behavior'] ) {
					$load_more_text = ! empty( $settings['tooto_mobile_load_more_text'] ) ? $settings['tooto_mobile_load_more_text'] : __( 'Load More', 'tooto-elementor' );
					echo '<div class="tooto-mobile-load-more-container" style="display:none;"><button class="tooto-mobile-load-more-btn">' . esc_html( $load_more_text ) . '</button></div>';
				}
			} else {
				// Fallback
				echo $content;
			}
		}

		public function render() {
			// render() 可能不会被执行（如果使用了 Skin），保留此空实现或调用父类即可
			parent::render();
		}
	
		protected function render_custom_pagination() {
			// Try to retrieve the query
			// 1. Check captured query from loop_start
			$query = $this->captured_query;

			// 2. Fallback to $this->query if available
			if ( ! $query && property_exists( $this, 'query' ) ) {
				$query = $this->query;
			} 
			
			// 3. Fallback to global query (least reliable after render)
			if ( ! $query && isset( $GLOBALS['wp_query'] ) ) {
				$query = $GLOBALS['wp_query'];
			}

			if ( ! $query || ! isset( $query->max_num_pages ) ) {
				return;
			}
	
			$total_pages = $query->max_num_pages;
			if ( $total_pages < 2 ) {
				return;
			}
	
			$current_page = ( get_query_var( 'paged' ) ) ? get_query_var( 'paged' ) : 1;
			// Handle Elementor preview or non-standard queries if needed
			if ( isset( $query->query_vars['paged'] ) && $query->query_vars['paged'] > 0 ) {
				$current_page = $query->query_vars['paged'];
			}
	
			$settings = $this->get_settings_for_display();
			
			// Prepare Icons
			ob_start();
			Icons_Manager::render_icon( $settings['tooto_pagination_prev_icon'], [ 'aria-hidden' => 'true' ] );
			$prev_icon = ob_get_clean();

			ob_start();
			Icons_Manager::render_icon( $settings['tooto_pagination_next_icon'], [ 'aria-hidden' => 'true' ] );
			$next_icon = ob_get_clean();

			// Fallback if icon rendering failed or empty
			if ( empty( $prev_icon ) ) $prev_icon = '<';
			if ( empty( $next_icon ) ) $next_icon = '>';

			echo '<div class="tooto-pagination">';
			
			// Standard Pagination (Numbers + Prev/Next)
			$big = 999999999; // need an unlikely integer
			echo paginate_links( [
				'base' => str_replace( $big, '%#%', esc_url( get_pagenum_link( $big ) ) ),
				'format' => '?paged=%#%',
				'current' => max( 1, $current_page ),
				'total' => $total_pages,
				'prev_text' => $prev_icon,
				'next_text' => $next_icon,
			] );
	
			// Custom "Jump To" Dropdown
			echo '<div class="tooto-pagination-jump">';
			echo '<span class="tooto-pagination-text">' . esc_html__( 'Page', 'tooto-elementor' ) . '</span>';
			
			// Select Wrapper
			echo '<div class="tooto-pagination-select-wrapper">';
			echo '<select class="tooto-pagination-select" onchange="window.location.href=this.value">';
			
			for ( $i = 1; $i <= $total_pages; $i++ ) {
				$link = get_pagenum_link( $i );
				$selected = ( $i == $current_page ) ? 'selected' : '';
				echo '<option value="' . esc_url( $link ) . '" ' . $selected . '>' . $i . '</option>';
			}
			
			echo '</select>';
			
			// Custom SVG Arrow
			echo '<span class="tooto-select-arrow">';
			echo '<svg xmlns="http://www.w3.org/2000/svg" xmlns:xlink="http://www.w3.org/1999/xlink" fill="none" version="1.1" width="10.5" height="6" viewBox="0 0 10.5 6"><path d="M11.542958,11.249370599999999L16.3734851,6.75041831C16.5421662,6.57780892,16.5421662,6.30017826,16.3734851,6.12838981C16.2049651,5.957203373,15.9300008,5.957203373,15.7629299,6.12838981L10.62208095,10.9169035C10.53312644,11.0082431,10.4948191135,11.1301203,10.50055996305,11.249316199999999C10.4948193179,11.369551699999999,10.533072818,11.4905534,10.62208095,11.583534199999999L15.7629299,16.369968C15.9300008,16.543343999999998,16.2049651,16.543343999999998,16.3734851,16.369968C16.542172,16.198859,16.542172,15.9204178,16.3734851,15.7493076L11.542958,11.249370599999999Z" fill="currentColor" fill-opacity="1" transform="matrix(0,-1,-1,0,16.5,16.5)"/></svg>';
			echo '</span>';
			
			echo '</div>'; // .tooto-pagination-select-wrapper

			echo '<span class="tooto-pagination-text">' . sprintf( esc_html__( 'Of %d', 'tooto-elementor' ), $total_pages ) . '</span>';
			echo '</div>'; // .tooto-pagination-jump
			
			echo '</div>'; // .tooto-pagination
		}
	}
} else {
	// 如果站点没有 Pro/Pro-Elements 的 Loop Grid，先不提供降级实现，以免控件不一致。
	//（需要的话可再加一个轻量 fallback）
	class Loop_Grid extends \Elementor\Widget_Base {
		public function get_name(): string {
			return 'tooto-loop-grid';
		}

		public function get_title(): string {
			return __( 'Tooto Loop Grid', 'tooto-elementor' );
		}

		public function get_categories(): array {
			return [ 'tooto' ];
		}

		public function get_icon(): string {
			return 'eicon-posts-grid';
		}

		public function get_style_depends(): array {
			return [ 'tooto-elementor' ];
		}

		protected function register_controls() {
			// Minimal controls if needed
		}

		protected function render(): void {
			// FORCE DEBUG OUTPUT (VISIBLE)
			echo '<div style="background:red; color:white; padding:20px; border:5px solid yellow; position:relative; z-index:99999; font-size:16px; line-height:1.5;">';
			echo '<strong>⚠️ TOOTO LOOP GRID DEBUG ⚠️</strong><br>';
			echo 'Class: Loop_Grid (Fallback Mode)<br>';
			echo 'Reason: ElementorPro\Modules\LoopBuilder\Widgets\Loop_Grid not found<br>';
			echo '</div>';

			echo '<div class="tooto-loop-grid__empty">' . esc_html__( '该组件需要 Elementor Pro（或 Pro Elements）提供 Loop Grid 功能。', 'tooto-elementor' ) . '</div>';
		}
	}
}
