<?php
/**
 * Tooto Post 组件：文章卡片，默认显示日期和标题，hover后显示摘要和readmore链接。
 */

namespace Tooto_Elementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;

class Post extends Widget_Base {
	public function get_name(): string {
		return 'tooto-post';
	}

	public function get_title(): string {
		return __( 'Tooto Post', 'tooto-elementor' );
	}

	public function get_icon(): string {
		return 'eicon-post';
	}

	public function get_categories(): array {
		return [ 'tooto' ];
	}

	public function get_keywords(): array {
		return [ 'post', 'article', 'card', '文章', '卡片', 'tooto' ];
	}

	public function get_style_depends(): array {
		return [ 'tooto-elementor' ];
	}

	protected function register_controls(): void {
		$this->start_controls_section(
			'section_content',
			[
				'label' => __( '内容', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'post_id',
			[
				'label'       => __( '选择文章', 'tooto-elementor' ),
				'type'        => Controls_Manager::SELECT2,
				'options'     => $this->get_posts_list(),
				'default'     => '',
				'label_block' => true,
			]
		);

		$this->add_control(
			'read_more_text',
			[
				'label'       => __( 'Read More 文字', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => __( 'Read More', 'tooto-elementor' ),
				'placeholder' => __( 'Read More', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'excerpt_length',
			[
				'label'   => __( '摘要长度', 'tooto-elementor' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 20,
				'min'     => 0,
				'max'     => 200,
				'step'    => 1,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style',
			[
				'label' => __( '样式', 'tooto-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'card_height',
			[
				'label'      => __( '卡片高度', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'vh' ],
				'range'      => [
					'px' => [
						'min' => 200,
						'max' => 800,
					],
					'vh' => [
						'min' => 20,
						'max' => 100,
					],
				],
				'default'    => [
					'size' => 400,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-post' => 'min-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'overlay_color',
			[
				'label'     => __( '遮罩颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0, 0, 0, 0.4)',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__overlay' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'overlay_hover_color',
			[
				'label'     => __( 'Hover 遮罩颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0, 0, 0, 0.6)',
				'selectors' => [
					'{{WRAPPER}} .tooto-post:hover .tooto-post__overlay' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_typography',
			[
				'label' => __( '文字样式', 'tooto-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'date_color',
			[
				'label'     => __( '日期颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__date' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'date_typography',
				'label'    => __( '日期字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__date',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => __( '标题颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__title' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tooto-post__title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => __( '标题字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__title',
			]
		);

		$this->add_control(
			'excerpt_color',
			[
				'label'     => __( '摘要颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__excerpt' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'excerpt_typography',
				'label'    => __( '摘要字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__excerpt',
			]
		);

		$this->add_control(
			'read_more_color',
			[
				'label'     => __( 'Read More 颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__read-more' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'read_more_typography',
				'label'    => __( 'Read More 字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__read-more',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * 获取文章列表
	 *
	 * @return array
	 */
	private function get_posts_list(): array {
		$posts = get_posts(
			[
				'numberposts' => -1,
				'post_status' => 'publish',
			]
		);

		$options = [];
		foreach ( $posts as $post ) {
			$options[ $post->ID ] = $post->post_title;
		}

		return $options;
	}

	protected function render(): void {
		$settings = $this->get_settings_for_display();

		if ( empty( $settings['post_id'] ) ) {
			echo '<div class="tooto-post__empty">' . esc_html__( '请选择一个文章', 'tooto-elementor' ) . '</div>';
			return;
		}

		$post_id = (int) $settings['post_id'];
		$post    = get_post( $post_id );

		if ( ! $post ) {
			echo '<div class="tooto-post__empty">' . esc_html__( '文章不存在', 'tooto-elementor' ) . '</div>';
			return;
		}

		// 设置全局post对象
		global $wp_query;
		$original_post = $wp_query->post;
		$wp_query->post = $post;
		setup_postdata( $post );

		$thumbnail_id = get_post_thumbnail_id( $post_id );
		$thumbnail_url = $thumbnail_id ? wp_get_attachment_image_url( $thumbnail_id, 'large' ) : '';
		$permalink = get_permalink( $post_id );
		$date = get_the_date( 'F d Y', $post_id );
		$title = get_the_title( $post_id );
		$excerpt_length = isset( $settings['excerpt_length'] ) ? (int) $settings['excerpt_length'] : 20;
		$excerpt = $this->get_excerpt( $post, $excerpt_length );
		$read_more_text = ! empty( $settings['read_more_text'] ) ? $settings['read_more_text'] : __( 'Read More', 'tooto-elementor' );

		?>
		<article class="tooto-post">
			<?php if ( $thumbnail_id ) : ?>
				<div class="tooto-post__thumbnail">
					<?php echo wp_get_attachment_image( $thumbnail_id, 'large', false, [ 'alt' => esc_attr( $title ) ] ); ?>
				</div>
			<?php endif; ?>
			<div class="tooto-post__overlay"></div>
			<div class="tooto-post__content">
				<div class="tooto-post__meta">
					<span class="tooto-post__date"><?php echo esc_html( $date ); ?></span>
				</div>
				<h3 class="tooto-post__title">
					<a href="<?php echo esc_url( $permalink ); ?>"><?php echo esc_html( $title ); ?></a>
				</h3>
				<div class="tooto-post__hover-content">
					<div class="tooto-post__hover-content__wrapper">
						<?php if ( $excerpt ) : ?>
							<div class="tooto-post__excerpt-wrapper">
								<div class="tooto-post__excerpt"><?php echo wp_kses_post( $excerpt ); ?></div>
							</div>
						<?php endif; ?>
						<a href="<?php echo esc_url( $permalink ); ?>" class="tooto-post__read-more">
							<?php echo esc_html( $read_more_text ); ?>
						</a>
					</div>
				</div>
			</div>
		</article>
		<?php

		// 恢复原始post对象
		$wp_query->post = $original_post;
		wp_reset_postdata();
	}

	/**
	 * 获取文章摘要
	 *
	 * @param \WP_Post $post 文章对象
	 * @param int      $length 摘要长度
	 * @return string
	 */
	private function get_excerpt( $post, int $length = 20 ): string {
		if ( ! empty( $post->post_excerpt ) ) {
			$excerpt = $post->post_excerpt;
		} else {
			$excerpt = $post->post_content;
		}

		$excerpt = strip_shortcodes( $excerpt );
		$excerpt = wp_strip_all_tags( $excerpt );
		$excerpt = wp_trim_words( $excerpt, $length, '...' );

		return $excerpt;
	}
}

