<?php
/**
 * Tooto Posts 组件：文章列表，支持配置展示数量和封面图片比例。
 */

namespace Tooto_Elementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;

class Posts extends Widget_Base {
	public function get_name(): string {
		return 'tooto-posts';
	}

	public function get_title(): string {
		return __( 'Tooto Posts', 'tooto-elementor' );
	}

	public function get_icon(): string {
		return 'eicon-posts-grid';
	}

	public function get_categories(): array {
		return [ 'tooto' ];
	}

	public function get_keywords(): array {
		return [ 'posts', 'articles', 'list', 'grid', '文章列表', 'tooto' ];
	}

	public function get_style_depends(): array {
		return [ 'tooto-elementor' ];
	}

	protected function register_controls(): void {
		$this->start_controls_section(
			'section_query',
			[
				'label' => __( '查询设置', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'posts_per_page',
			[
				'label'   => __( '文章数量', 'tooto-elementor' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 6,
				'min'     => 1,
				'max'     => 50,
				'step'    => 1,
			]
		);

		$this->add_control(
			'orderby',
			[
				'label'   => __( '排序方式', 'tooto-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'date',
				'options' => [
					'date'       => __( '日期', 'tooto-elementor' ),
					'title'      => __( '标题', 'tooto-elementor' ),
					'menu_order' => __( '菜单顺序', 'tooto-elementor' ),
					'rand'       => __( '随机', 'tooto-elementor' ),
				],
			]
		);

		$this->add_control(
			'order',
			[
				'label'   => __( '排序方向', 'tooto-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'DESC',
				'options' => [
					'ASC'  => __( '升序', 'tooto-elementor' ),
					'DESC' => __( '降序', 'tooto-elementor' ),
				],
			]
		);

		$this->add_control(
			'category',
			[
				'label'       => __( '分类', 'tooto-elementor' ),
				'type'        => Controls_Manager::SELECT2,
				'options'     => $this->get_categories_list(),
				'multiple'    => true,
				'label_block' => true,
				'default'     => [],
			]
		);

		$this->add_control(
			'pagination',
			[
				'label'        => __( '启用分页', 'tooto-elementor' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => __( '是', 'tooto-elementor' ),
				'label_off'    => __( '否', 'tooto-elementor' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_layout',
			[
				'label' => __( '布局设置', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'columns',
			[
				'label'   => __( '列数', 'tooto-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'default' => '3',
				'options' => [
					'1' => '1',
					'2' => '2',
					'3' => '3',
					'4' => '4',
				],
			]
		);

		$this->add_control(
			'image_aspect_ratio',
			[
				'label'   => __( '封面图片比例', 'tooto-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'default' => 'landscape',
				'options' => [
					'landscape' => __( '横向 (16:9)', 'tooto-elementor' ),
					'portrait'  => __( '竖向 (9:16)', 'tooto-elementor' ),
					'square'    => __( '正方形 (1:1)', 'tooto-elementor' ),
					'auto'      => __( '自动', 'tooto-elementor' ),
				],
			]
		);

		$this->add_control(
			'gap',
			[
				'label'      => __( '间距', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 60,
					],
				],
				'default'    => [
					'size' => 24,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-posts__grid' => 'gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_content',
			[
				'label' => __( '内容设置', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'excerpt_length',
			[
				'label'   => __( '摘要长度', 'tooto-elementor' ),
				'type'    => Controls_Manager::NUMBER,
				'default' => 20,
				'min'     => 0,
				'max'     => 200,
				'step'    => 1,
			]
		);

		$this->add_control(
			'read_more_text',
			[
				'label'       => __( 'Read More 文字', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => __( 'Read More', 'tooto-elementor' ),
				'placeholder' => __( 'Read More', 'tooto-elementor' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style',
			[
				'label' => __( '样式', 'tooto-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'card_height',
			[
				'label'      => __( '卡片高度', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'vh' ],
				'range'      => [
					'px' => [
						'min' => 200,
						'max' => 800,
					],
					'vh' => [
						'min' => 20,
						'max' => 100,
					],
				],
				'default'    => [
					'size' => 400,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-post' => 'min-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'overlay_color',
			[
				'label'     => __( '遮罩颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0, 0, 0, 0.4)',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__overlay' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'overlay_hover_color',
			[
				'label'     => __( 'Hover 遮罩颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0, 0, 0, 0.6)',
				'selectors' => [
					'{{WRAPPER}} .tooto-post:hover .tooto-post__overlay' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_typography',
			[
				'label' => __( '文字样式', 'tooto-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_control(
			'date_color',
			[
				'label'     => __( '日期颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__date' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'date_typography',
				'label'    => __( '日期字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__date',
			]
		);

		$this->add_control(
			'title_color',
			[
				'label'     => __( '标题颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__title' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tooto-post__title a' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'title_typography',
				'label'    => __( '标题字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__title',
			]
		);

		$this->add_control(
			'excerpt_color',
			[
				'label'     => __( '摘要颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__excerpt' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'excerpt_typography',
				'label'    => __( '摘要字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__excerpt',
			]
		);

		$this->add_control(
			'read_more_color',
			[
				'label'     => __( 'Read More 颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-post__read-more' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'read_more_typography',
				'label'    => __( 'Read More 字体', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-post__read-more',
			]
		);

		$this->end_controls_section();
	}

	/**
	 * 获取分类列表
	 *
	 * @return array
	 */
	private function get_categories_list(): array {
		$categories = get_categories( [ 'hide_empty' => false ] );
		$options    = [];

		foreach ( $categories as $category ) {
			$options[ $category->term_id ] = $category->name;
		}

		return $options;
	}

	/**
	 * 查询文章
	 *
	 * @param array $settings 设置
	 * @return \WP_Query
	 */
	private function query_posts( array $settings ): \WP_Query {
		$paged = get_query_var( 'paged' ) ? get_query_var( 'paged' ) : 1;
		if ( isset( $_GET['tooto_page'] ) ) {
			$paged = absint( $_GET['tooto_page'] );
		}

		$args = [
			'post_type'      => 'post',
			'post_status'    => 'publish',
			'posts_per_page' => isset( $settings['posts_per_page'] ) ? (int) $settings['posts_per_page'] : 6,
			'orderby'        => isset( $settings['orderby'] ) ? $settings['orderby'] : 'date',
			'order'          => isset( $settings['order'] ) ? $settings['order'] : 'DESC',
			'paged'          => $paged,
		];

		if ( ! empty( $settings['category'] ) && is_array( $settings['category'] ) ) {
			$args['category__in'] = array_map( 'intval', $settings['category'] );
		}

		return new \WP_Query( $args );
	}

	protected function render(): void {
		$settings = $this->get_settings_for_display();
		$query    = $this->query_posts( $settings );

		if ( ! $query->have_posts() ) {
			echo '<div class="tooto-posts__empty">' . esc_html__( '没有找到文章', 'tooto-elementor' ) . '</div>';
			return;
		}

		$columns          = isset( $settings['columns'] ) ? $settings['columns'] : '3';
		$image_aspect     = isset( $settings['image_aspect_ratio'] ) ? $settings['image_aspect_ratio'] : 'landscape';
		$excerpt_length   = isset( $settings['excerpt_length'] ) ? (int) $settings['excerpt_length'] : 20;
		$read_more_text   = ! empty( $settings['read_more_text'] ) ? $settings['read_more_text'] : __( 'Read More', 'tooto-elementor' );
		$aspect_class     = 'tooto-post--aspect-' . esc_attr( $image_aspect );

		?>
		<div class="tooto-posts__grid tooto-posts__grid--columns-<?php echo esc_attr( $columns ); ?>">
			<?php
			while ( $query->have_posts() ) {
				$query->the_post();

				$thumbnail_id  = get_post_thumbnail_id();
				$thumbnail_url = $thumbnail_id ? wp_get_attachment_image_url( $thumbnail_id, 'large' ) : '';
				$permalink     = get_permalink();
				$date          = get_the_date( 'F d Y' );
				$title         = get_the_title();
				$excerpt       = $this->get_excerpt( get_the_excerpt() ?: get_the_content(), $excerpt_length );

				?>
				<article class="tooto-post <?php echo esc_attr( $aspect_class ); ?>">
					<?php if ( $thumbnail_id ) : ?>
						<div class="tooto-post__thumbnail">
							<?php echo wp_get_attachment_image( $thumbnail_id, 'large', false, [ 'alt' => esc_attr( $title ) ] ); ?>
						</div>
					<?php endif; ?>
					<div class="tooto-post__overlay"></div>
					<div class="tooto-post__content">
						<div class="tooto-post__meta">
							<span class="tooto-post__date"><?php echo esc_html( $date ); ?></span>
						</div>
						<h3 class="tooto-post__title">
							<a href="<?php echo esc_url( $permalink ); ?>"><?php echo esc_html( $title ); ?></a>
						</h3>
						<div class="tooto-post__hover-content">
							<div class="tooto-post__hover-content__wrapper">
								<?php if ( $excerpt ) : ?>
									<div class="tooto-post__excerpt-wrapper">
										<div class="tooto-post__excerpt"><?php echo wp_kses_post( $excerpt ); ?></div>
									</div>
								<?php endif; ?>
								<a href="<?php echo esc_url( $permalink ); ?>" class="tooto-post__read-more">
									<?php echo esc_html( $read_more_text ); ?>
								</a>
							</div>
						</div>
					</div>
				</article>
				<?php
			}
			wp_reset_postdata();
			?>
		</div>
		<?php
		if ( 'yes' === ( $settings['pagination'] ?? 'yes' ) && $query->max_num_pages > 1 ) {
			$this->render_pagination( $query, $settings );
		}
	}

	/**
	 * 渲染分页
	 *
	 * @param \WP_Query $query 查询对象
	 * @param array     $settings 设置
	 * @return void
	 */
	private function render_pagination( \WP_Query $query, array $settings ): void {
		$current_page = max( 1, get_query_var( 'paged' ) );
		if ( isset( $_GET['tooto_page'] ) ) {
			$current_page = absint( $_GET['tooto_page'] );
		}
		$total_pages = $query->max_num_pages;
		
		// 构建基础 URL
		$request_uri = isset( $_SERVER['REQUEST_URI'] ) ? esc_url_raw( wp_unslash( $_SERVER['REQUEST_URI'] ) ) : '';
		$parsed_url  = wp_parse_url( $request_uri );
		$base_url    = ( isset( $parsed_url['scheme'] ) ? $parsed_url['scheme'] . '://' : '' ) .
					   ( isset( $parsed_url['host'] ) ? $parsed_url['host'] : '' ) .
					   ( isset( $parsed_url['path'] ) ? $parsed_url['path'] : '' );
		
		// 移除分页参数
		$base_url = remove_query_arg( [ 'tooto_page', 'paged' ], $base_url );

		// 计算显示的页码范围
		$page_limit = 5;
		$start_page = max( 1, $current_page - floor( $page_limit / 2 ) );
		$end_page   = min( $total_pages, $start_page + $page_limit - 1 );
		if ( $end_page - $start_page < $page_limit - 1 ) {
			$start_page = max( 1, $end_page - $page_limit + 1 );
		}

		$prev_url = $current_page > 1 ? add_query_arg( 'tooto_page', $current_page - 1, $base_url ) : '#';
		$next_url = $current_page < $total_pages ? add_query_arg( 'tooto_page', $current_page + 1, $base_url ) : '#';

		?>
		<nav class="tooto-posts__pagination" aria-label="<?php esc_attr_e( '文章分页', 'tooto-elementor' ); ?>">
			<div class="tooto-posts__pagination-nav">
				<a href="<?php echo esc_url( $prev_url ); ?>" class="tooto-posts__pagination-arrow tooto-posts__pagination-arrow--prev <?php echo $current_page <= 1 ? 'is-disabled' : ''; ?>" <?php echo $current_page <= 1 ? 'aria-disabled="true" tabindex="-1"' : ''; ?>>
					<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" aria-hidden="true">
						<path d="M10 12L6 8L10 4" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
					</svg>
					<span class="screen-reader-text"><?php esc_html_e( '上一页', 'tooto-elementor' ); ?></span>
				</a>

				<div class="tooto-posts__pagination-numbers">
					<?php
					for ( $i = $start_page; $i <= $end_page; $i++ ) {
						$page_url = add_query_arg( 'tooto_page', $i, $base_url );
						$is_active = $i === $current_page;
						?>
						<a href="<?php echo esc_url( $page_url ); ?>" class="tooto-posts__pagination-number <?php echo $is_active ? 'is-active' : ''; ?>" <?php echo $is_active ? 'aria-current="page"' : ''; ?>>
							<?php echo esc_html( $i ); ?>
						</a>
						<?php
					}
					?>
				</div>

				<a href="<?php echo esc_url( $next_url ); ?>" class="tooto-posts__pagination-arrow tooto-posts__pagination-arrow--next <?php echo $current_page >= $total_pages ? 'is-disabled' : ''; ?>" <?php echo $current_page >= $total_pages ? 'aria-disabled="true" tabindex="-1"' : ''; ?>>
					<svg width="16" height="16" viewBox="0 0 16 16" fill="none" xmlns="http://www.w3.org/2000/svg" aria-hidden="true">
						<path d="M6 4L10 8L6 12" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
					</svg>
					<span class="screen-reader-text"><?php esc_html_e( '下一页', 'tooto-elementor' ); ?></span>
				</a>
			</div>

			<div class="tooto-posts__pagination-jump">
				<span class="tooto-posts__pagination-label"><?php esc_html_e( 'Page', 'tooto-elementor' ); ?></span>
				<div class="tooto-posts__pagination-dropdown">
					<button type="button" class="tooto-posts__pagination-dropdown-toggle" aria-label="<?php esc_attr_e( '跳转到页面', 'tooto-elementor' ); ?>" aria-expanded="false">
						<span class="tooto-posts__pagination-dropdown-value"><?php echo esc_html( $current_page ); ?></span>
						<svg class="tooto-posts__pagination-dropdown-icon" width="12" height="12" viewBox="0 0 12 12" fill="none" xmlns="http://www.w3.org/2000/svg" aria-hidden="true">
							<path d="M3 4.5L6 7.5L9 4.5" stroke="currentColor" stroke-width="1.5" stroke-linecap="round" stroke-linejoin="round"/>
						</svg>
					</button>
					<ul class="tooto-posts__pagination-dropdown-menu" role="listbox">
						<?php
						for ( $i = 1; $i <= $total_pages; $i++ ) {
							$is_selected = $i === $current_page;
							$page_url = add_query_arg( 'tooto_page', $i, $base_url );
							?>
							<li role="option" aria-selected="<?php echo $is_selected ? 'true' : 'false'; ?>">
								<a href="<?php echo esc_url( $page_url ); ?>" class="tooto-posts__pagination-dropdown-item <?php echo $is_selected ? 'is-selected' : ''; ?>" data-value="<?php echo esc_attr( $i ); ?>">
									<?php echo esc_html( $i ); ?>
								</a>
							</li>
							<?php
						}
						?>
					</ul>
				</div>
				<span class="tooto-posts__pagination-label"><?php printf( esc_html__( 'Of %d', 'tooto-elementor' ), $total_pages ); ?></span>
			</div>
		</nav>
		<?php
	}

	/**
	 * 获取文章摘要
	 *
	 * @param string $content 内容
	 * @param int    $length 摘要长度
	 * @return string
	 */
	private function get_excerpt( string $content, int $length = 20 ): string {
		$excerpt = strip_shortcodes( $content );
		$excerpt = wp_strip_all_tags( $excerpt );
		$excerpt = wp_trim_words( $excerpt, $length, '...' );

		return $excerpt;
	}
}

