<?php
/**
 * Tooto Solutions Tabs 组件：展示 solutions 文章 + 关联产品卡片。
 */

namespace Tooto_Elementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;
use Elementor\Group_Control_Typography;
use Elementor\Widget_Base;
use Elementor\Icons_Manager;

class Solutions_Tabs extends Widget_Base {
	public function get_name(): string {
		return 'tooto-solutions-tabs';
	}

	public function get_title(): string {
		return __( 'Tooto Solutions Tabs', 'tooto-elementor' );
	}

	public function get_icon(): string {
		return 'eicon-tabs';
	}

	public function get_categories(): array {
		return [ 'tooto' ];
	}

	public function get_keywords(): array {
		return [ 'solutions', 'tab', 'products', '案例', '解决方案', '产品', 'tooto' ];
	}

	public function get_style_depends(): array {
		return [ 'tooto-elementor' ];
	}

	public function get_script_depends(): array {
		return [ 'tooto-elementor' ];
	}

	protected function register_controls(): void {
		$this->start_controls_section(
			'section_query',
			[
				'label' => __( 'Solutions 列表', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'post_type',
			[
				'label'       => __( 'Solutions 文章类型', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => 'solutions',
				'description' => __( 'SCF 创建的解决方案文章类型 slug。', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'posts_per_page',
			[
				'label'   => __( '显示数量', 'tooto-elementor' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 20,
				'default' => 8,
			]
		);

		$this->add_control(
			'orderby',
			[
				'label'   => __( '排序字段', 'tooto-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'date'       => __( '发布日期', 'tooto-elementor' ),
					'title'      => __( '标题', 'tooto-elementor' ),
					'menu_order' => __( '菜单顺序', 'tooto-elementor' ),
					'rand'       => __( '随机', 'tooto-elementor' ),
				],
				'default' => 'date',
			]
		);

		$this->add_control(
			'order',
			[
				'label'   => __( '顺序', 'tooto-elementor' ),
				'type'    => Controls_Manager::SELECT,
				'options' => [
					'DESC' => __( '倒序', 'tooto-elementor' ),
					'ASC'  => __( '正序', 'tooto-elementor' ),
				],
				'default' => 'DESC',
			]
		);

		$this->add_control(
			'taxonomy',
			[
				'label'       => __( 'Solutions 分类法（可选）', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => 'solution_category',
			]
		);

		$this->add_control(
			'term_slugs',
			[
				'label'       => __( '筛选分类 slug（逗号分隔，可选）', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'placeholder' => 'auto,energy',
				'description' => __( '留空则不过滤分类。', 'tooto-elementor' ),
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_products',
			[
				'label' => __( '关联产品', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'product_meta_key',
			[
				'label'       => __( '产品字段 meta key', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => 'related_products',
				'description' => __( 'SCF 多选产品字段的 meta key。需返回产品文章 ID 数组或逗号分隔字符串。', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'product_post_type',
			[
				'label'       => __( '产品文章类型', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => 'product',
				'description' => __( 'SCF 选择的产品文章类型 slug。', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'product_limit',
			[
				'label'   => __( '产品展示数量', 'tooto-elementor' ),
				'type'    => Controls_Manager::NUMBER,
				'min'     => 1,
				'max'     => 6,
				'default' => 3,
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_labels',
			[
				'label' => __( '显示与文案', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'button_text',
			[
				'label'       => __( '按钮文字', 'tooto-elementor' ),
				'type'        => Controls_Manager::TEXT,
				'default'     => __( 'More', 'tooto-elementor' ),
				'placeholder' => __( 'More', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'enable_arrows',
			[
				'label'        => __( '显示左右切换', 'tooto-elementor' ),
				'type'         => Controls_Manager::SWITCHER,
				'label_on'     => __( '显示', 'tooto-elementor' ),
				'label_off'    => __( '隐藏', 'tooto-elementor' ),
				'return_value' => 'yes',
				'default'      => 'yes',
			]
		);

		$this->add_control(
			'arrow_icon_prev',
			[
				'label'   => __( '上一张图标', 'tooto-elementor' ),
				'type'    => Controls_Manager::ICONS,
				'default' => [
					'value'   => 'eicon-chevron-left',
					'library' => 'elementor',
				],
				'condition' => [
					'enable_arrows' => 'yes',
				],
			]
		);

		$this->add_control(
			'arrow_icon_next',
			[
				'label'   => __( '下一张图标', 'tooto-elementor' ),
				'type'    => Controls_Manager::ICONS,
				'default' => [
					'value'   => 'eicon-chevron-right',
					'library' => 'elementor',
				],
				'condition' => [
					'enable_arrows' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'arrow_size',
			[
				'label'      => __( '箭头尺寸', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 12,
						'max' => 48,
					],
				],
				'default'    => [
					'size' => 18,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn' => 'font-size: {{SIZE}}{{UNIT}}; width: calc({{SIZE}}{{UNIT}} + 18px); height: calc({{SIZE}}{{UNIT}} + 18px);',
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn i' => 'font-size: {{SIZE}}{{UNIT}};',
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn svg' => 'width: {{SIZE}}{{UNIT}}; height: {{SIZE}}{{UNIT}};',
				],
				'condition'  => [
					'enable_arrows' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'arrow_offset_prev',
			[
				'label'      => __( '左箭头横向偏移', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => -200,
						'max' => 200,
					],
				],
				'default'    => [
					'size' => 0,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn[data-tooto-solutions-nav="prev"]' => 'transform: translateX({{SIZE}}{{UNIT}});',
				],
				'condition'  => [
					'enable_arrows' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'arrow_offset_next',
			[
				'label'      => __( '右箭头横向偏移', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => -200,
						'max' => 200,
					],
				],
				'default'    => [
					'size' => 0,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn[data-tooto-solutions-nav="next"]' => 'transform: translateX({{SIZE}}{{UNIT}});',
				],
				'condition'  => [
					'enable_arrows' => 'yes',
				],
			]
		);

		$this->add_control(
			'arrow_color',
			[
				'label'     => __( '箭头颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#ffffff',
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn' => 'color: {{VALUE}};',
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn svg' => 'fill: {{VALUE}};',
				],
				'condition' => [
					'enable_arrows' => 'yes',
				],
			]
		);

		$this->add_control(
			'arrow_bg_color',
			[
				'label'     => __( '箭头背景', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => 'rgba(0,0,0,0.55)',
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__nav-btn' => 'background: {{VALUE}};',
				],
				'condition' => [
					'enable_arrows' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_tabs',
			[
				'label' => __( 'Tab 样式', 'tooto-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'tab_typo',
				'label'    => __( '文字', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-solutions-tabs__tab',
			]
		);

		$this->add_control(
			'tab_color',
			[
				'label'     => __( '文字颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'tab_color_hover',
			[
				'label'     => __( '悬停文字颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'tab_active_color',
			[
				'label'     => __( '激活文字', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab.is-active' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'tab_underline_color',
			[
				'label'     => __( '激活下划线', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab.is-active::after' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'tab_typo_hover',
				'label'    => __( '悬停文字排版', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-solutions-tabs__tab:hover',
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'tab_typo_active',
				'label'    => __( '激活文字排版', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-solutions-tabs__tab.is-active',
			]
		);

		$this->add_responsive_control(
			'tab_gap',
			[
				'label'      => __( 'Tab 间距', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 48,
					],
				],
				'default'    => [
					'size' => 24,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__tabs' => 'column-gap: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tab_padding',
			[
				'label'      => __( 'Tab 内边距', 'tooto-elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', 'em' ],
				'default'    => [
					'top'    => '12',
					'right'  => '16',
					'bottom' => '12',
					'left'   => '16',
					'unit'   => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'tab_border_width',
			[
				'label'      => __( '边框宽度', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 8,
					],
				],
				'default'    => [
					'size' => 1,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab' => 'border-width: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->add_control(
			'tab_border_radius',
			[
				'label'      => __( '圆角', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px' ],
				'range'      => [
					'px' => [
						'min' => 0,
						'max' => 16,
					],
				],
				'default'    => [
					'size' => 0,
					'unit' => 'px',
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab' => 'border-radius: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->start_controls_tabs( 'tabs_states' );

		$this->start_controls_tab(
			'tabs_normal',
			[
				'label' => __( '正常', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'tab_bg',
			[
				'label'     => __( '背景色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'tab_border_color',
			[
				'label'     => __( '边框颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();

		$this->start_controls_tab(
			'tabs_active',
			[
				'label' => __( '激活', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'tab_bg_active',
			[
				'label'     => __( '激活背景', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'default'   => '#d7d9dd',
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab.is-active' => 'background-color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'tab_border_color_active',
			[
				'label'     => __( '激活边框色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__tab.is-active' => 'border-color: {{VALUE}};',
				],
			]
		);

		$this->end_controls_tab();
		$this->end_controls_tabs();

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_hero',
			[
				'label' => __( '轮播图', 'tooto-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_responsive_control(
			'hero_padding',
			[
				'label'      => __( '轮播图内边距', 'tooto-elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__hero' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'hero_margin',
			[
				'label'      => __( '轮播图外边距', 'tooto-elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__hero' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'hero_height',
			[
				'label'      => __( '高度', 'tooto-elementor' ),
				'type'       => Controls_Manager::SLIDER,
				'size_units' => [ 'px', 'vh' ],
				'range'      => [
					'px' => [
						'min' => 200,
						'max' => 1200,
					],
					'vh' => [
						'min' => 20,
						'max' => 100,
					],
				],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__hero' => 'height: {{SIZE}}{{UNIT}}; min-height: {{SIZE}}{{UNIT}};',
				],
			]
		);

		$this->end_controls_section();

		$this->start_controls_section(
			'section_style_products',
			[
				'label' => __( '关联产品区域', 'tooto-elementor' ),
				'tab'   => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'hero_title_typo',
				'label'    => __( '主标题排版', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-solutions-tabs__hero-title',
			]
		);

		$this->add_control(
			'hero_title_color',
			[
				'label'     => __( '主标题颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__hero-title' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'hero_more_typo',
				'label'    => __( 'More 按钮排版', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-solutions-tabs__hero-link',
			]
		);

		$this->add_control(
			'hero_more_color',
			[
				'label'     => __( 'More 颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__hero-link' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_control(
			'hero_more_color_hover',
			[
				'label'     => __( 'More 悬停颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__hero-link:hover' => 'color: {{VALUE}};',
				],
			]
		);

		$this->add_responsive_control(
			'products_padding',
			[
				'label'      => __( '产品区域内边距', 'tooto-elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__products' => 'padding: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_responsive_control(
			'products_margin',
			[
				'label'      => __( '产品区域外边距', 'tooto-elementor' ),
				'type'       => Controls_Manager::DIMENSIONS,
				'size_units' => [ 'px', '%' ],
				'selectors'  => [
					'{{WRAPPER}} .tooto-solutions-tabs__products' => 'margin: {{TOP}}{{UNIT}} {{RIGHT}}{{UNIT}} {{BOTTOM}}{{UNIT}} {{LEFT}}{{UNIT}};',
				],
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name'     => 'products_title_typo',
				'label'    => __( '产品标题排版', 'tooto-elementor' ),
				'selector' => '{{WRAPPER}} .tooto-solutions-tabs__product-title',
			]
		);

		$this->add_control(
			'products_title_color',
			[
				'label'     => __( '产品标题颜色', 'tooto-elementor' ),
				'type'      => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-solutions-tabs__product-title' => 'color: {{VALUE}};',
				],
			]
		);

		// 产品卡片已改为整卡可点击且不显示 More 文案，因此不再暴露“产品 More”相关样式项，避免无效配置。

		$this->end_controls_section();
	}

	protected function render(): void {
		$settings  = $this->get_settings_for_display();
		$solutions = $this->query_solutions( $settings );

		if ( empty( $solutions ) ) {
			echo '<div class="tooto-solutions-tabs__empty">' . esc_html__( '暂无解决方案数据', 'tooto-elementor' ) . '</div>';
			return;
		}

		$button_text   = ! empty( $settings['button_text'] ) ? $settings['button_text'] : __( 'More', 'tooto-elementor' );
		$show_arrows   = 'yes' === ( $settings['enable_arrows'] ?? 'yes' );
		$meta_key      = $settings['product_meta_key'] ?? 'related_products';
		$product_type  = $settings['product_post_type'] ?? 'product';
		$product_limit = absint( $settings['product_limit'] ?? 3 );

		$products_map = [];

		?>
		<div class="tooto-solutions-tabs" data-tooto-solutions-tabs>
			<div class="tooto-solutions-tabs__tabs" role="tablist">
				<?php foreach ( $solutions as $index => $item ) : ?>
					<?php
					$is_active   = 0 === $index;
					$tab_classes = 'tooto-solutions-tabs__tab' . ( $is_active ? ' is-active' : '' );
					?>
					<button
						class="<?php echo esc_attr( $tab_classes ); ?>"
						role="tab"
						aria-selected="<?php echo $is_active ? 'true' : 'false'; ?>"
						data-target="tooto-solution-<?php echo esc_attr( $item['id'] ); ?>"
						type="button"
					>
						<?php echo esc_html( $item['title'] ); ?>
					</button>
				<?php endforeach; ?>
			</div>

			<div class="tooto-solutions-tabs__panels">
				<div class="tooto-solutions-tabs__swiper swiper">
					<div class="swiper-wrapper">
					<?php foreach ( $solutions as $index => $item ) : ?>
						<?php
						$is_active  = 0 === $index;
						$panel_id   = 'tooto-solution-' . $item['id'];
						$products   = $this->query_products( $item['id'], $meta_key, $product_type, $product_limit );
						$slide_cls  = 'tooto-solutions-tabs__panel swiper-slide' . ( $is_active ? ' is-active' : '' );
						$products_map[ $index ] = $products;
						?>
						<div
							id="<?php echo esc_attr( $panel_id ); ?>"
							class="<?php echo esc_attr( $slide_cls ); ?>"
							role="tabpanel"
							aria-hidden="<?php echo $is_active ? 'false' : 'true'; ?>"
						>
							<div class="tooto-solutions-tabs__hero">
								<div class="tooto-solutions-tabs__hero-media">
									<img src="<?php echo esc_url( $item['image'] ); ?>" alt="<?php echo esc_attr( $item['title'] ); ?>" loading="lazy" />
								</div>
								<div class="tooto-solutions-tabs__hero-content">
									<h3 class="tooto-solutions-tabs__hero-title"><?php echo esc_html( $item['title'] ); ?></h3>
									<?php if ( $item['permalink'] ) : ?>
										<a class="tooto-solutions-tabs__hero-link" href="<?php echo esc_url( $item['permalink'] ); ?>">
											<?php echo esc_html( $button_text ); ?>
										</a>
									<?php endif; ?>
								</div>
							</div>
						</div>
					<?php endforeach; ?>
					</div>
				</div>

				<?php if ( $show_arrows ) : ?>
					<div class="tooto-solutions-tabs__nav">
						<button class="tooto-solutions-tabs__nav-btn" data-tooto-solutions-nav="prev" type="button" aria-label="<?php esc_attr_e( '上一项', 'tooto-elementor' ); ?>">
							<?php Icons_Manager::render_icon( $settings['arrow_icon_prev'], [ 'aria-hidden' => 'true' ] ); ?>
						</button>
						<button class="tooto-solutions-tabs__nav-btn" data-tooto-solutions-nav="next" type="button" aria-label="<?php esc_attr_e( '下一项', 'tooto-elementor' ); ?>">
							<?php Icons_Manager::render_icon( $settings['arrow_icon_next'], [ 'aria-hidden' => 'true' ] ); ?>
						</button>
					</div>
				<?php endif; ?>
			</div>

			<div
				class="tooto-solutions-tabs__products"
				data-tooto-products
				data-products="<?php echo esc_attr( wp_json_encode( $products_map ) ); ?>"
			>
				<?php
				if ( ! empty( $products_map[0] ) ) {
					$this->render_products( $products_map[0] );
				}
				?>
			</div>
		</div>
		<?php
	}

	/**
	 * 查询 solutions 数据。
	 *
	 * @param array $settings 控件设置。
	 *
	 * @return array<int,array{id:int,title:string,permalink:string,image:string}>
	 */
	private function query_solutions( array $settings ): array {
		$post_type      = $settings['post_type'] ?? 'solutions';
		$posts_per_page = absint( $settings['posts_per_page'] ?? 8 );
		$orderby        = $settings['orderby'] ?? 'date';
		$order          = $settings['order'] ?? 'DESC';
		$taxonomy       = $settings['taxonomy'] ?? '';
		$term_slugs     = $this->parse_csv( $settings['term_slugs'] ?? '' );

		$args = [
			'post_type'      => $post_type,
			'post_status'    => 'publish',
			'posts_per_page' => $posts_per_page,
			'orderby'        => $orderby,
			'order'          => $order,
		];

		if ( $taxonomy && ! empty( $term_slugs ) ) {
			$args['tax_query'] = [
				[
					'taxonomy' => $taxonomy,
					'field'    => 'slug',
					'terms'    => $term_slugs,
				],
			];
		}

		$query   = new \WP_Query( $args );
		$results = [];

		while ( $query->have_posts() ) {
			$query->the_post();
			$post_id = get_the_ID();

			$results[] = [
				'id'        => $post_id,
				'title'     => get_the_title( $post_id ),
				'permalink' => get_permalink( $post_id ),
				'image'     => get_the_post_thumbnail_url( $post_id, 'large' ) ?: '',
			];
		}

		wp_reset_postdata();

		return $results;
	}

	/**
	 * 查询关联产品。
	 *
	 * @param int    $solution_id   方案文章 ID。
	 * @param string $meta_key      SCF meta key。
	 * @param string $post_type     产品 post type。
	 * @param int    $limit         展示数量。
	 *
	 * @return array<int,array{title:string,thumb:string,permalink:string}>
	 */
	private function query_products( int $solution_id, string $meta_key, string $post_type, int $limit ): array {
		$raw = get_post_meta( $solution_id, $meta_key, true );
		$ids = $this->normalize_ids( $raw );

		if ( empty( $ids ) ) {
			return [];
		}

		$ids = array_slice( $ids, 0, $limit );

		$posts = get_posts(
			[
				'post_type'      => $post_type,
				'post__in'       => $ids,
				'orderby'        => 'post__in',
				'posts_per_page' => $limit,
				'post_status'    => 'publish',
			]
		);

		$products = [];
		foreach ( $posts as $post ) {
			$products[] = [
				'title'     => get_the_title( $post ),
				'thumb'     => get_the_post_thumbnail_url( $post, 'medium' ) ?: '',
				'permalink' => get_permalink( $post ),
			];
		}

		return $products;
	}

	/**
	 * 将 meta 值转为 ID 数组。
	 *
	 * @param mixed $value 原始 meta。
	 *
	 * @return array<int>
	 */
	private function normalize_ids( $value ): array {
		if ( empty( $value ) ) {
			return [];
		}

		if ( is_array( $value ) ) {
			$flat = [];
			foreach ( $value as $v ) {
				if ( is_numeric( $v ) ) {
					$flat[] = (int) $v;
				}
			}
			return array_values( array_unique( $flat ) );
		}

		if ( is_string( $value ) ) {
			return $this->parse_csv( $value, true );
		}

		return [];
	}

	/**
	 * 解析逗号分隔值。
	 *
	 * @param string $csv      输入字符串。
	 * @param bool   $as_int   是否转为 int。
	 *
	 * @return array<int|string>
	 */
	private function parse_csv( string $csv, bool $as_int = false ): array {
		$csv = trim( $csv );
		if ( '' === $csv ) {
			return [];
		}

		$parts = array_map( 'trim', explode( ',', $csv ) );
		$parts = array_filter( $parts );

		if ( $as_int ) {
			return array_values(
				array_unique(
					array_map(
						static function ( $item ) {
							return (int) $item;
						},
						$parts
					)
				)
			);
		}

		return array_values( array_unique( $parts ) );
	}

	/**
	 * 输出产品卡片列表。
	 *
	 * @param array<int,array{title:string,thumb:string,permalink:string}> $products 产品数据。
	 */
	private function render_products( array $products ): void {
		if ( empty( $products ) ) {
			return;
		}

		echo '<div class="tooto-solutions-tabs__products-grid">';

		foreach ( $products as $product ) {
			echo '<div class="tooto-solutions-tabs__product-card">';

			if ( ! empty( $product['thumb'] ) ) {
				echo '<div class="tooto-solutions-tabs__product-thumb">';
				echo '<img src="' . esc_url( $product['thumb'] ) . '" alt="' . esc_attr( $product['title'] ) . '" loading="lazy" />';
				echo '</div>';
			}

			echo '<div class="tooto-solutions-tabs__product-body">';
			echo '<div class="tooto-solutions-tabs__product-title">' . esc_html( $product['title'] ) . '</div>';
			echo '</div>';

			if ( ! empty( $product['permalink'] ) ) {
				// 空链接：用伪元素 inset:0 扩大到整个卡片，且不显示/不占位文本。
				echo '<a class="tooto-solutions-tabs__product-link" href="' . esc_url( $product['permalink'] ) . '" aria-label="' . esc_attr( $product['title'] ) . '"></a>';
			}

			echo '</div>';
		}

		echo '</div>';
	}
}

