<?php
/**
 * Tooto Text Type Widget
 */

namespace Tooto_Elementor\Widgets;

if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Elementor\Controls_Manager;
use Elementor\Widget_Base;
use Elementor\Group_Control_Typography;
use Elementor\Group_Control_Text_Shadow;

class Text_Type extends Widget_Base {
	public function get_name(): string {
		return 'tooto-text-type';
	}

	public function get_title(): string {
		return __( 'Tooto 打字效果', 'tooto-elementor' );
	}

	public function get_icon(): string {
		return 'eicon-animation-text';
	}

	public function get_categories(): array {
		return [ 'tooto' ];
	}

	public function get_keywords(): array {
		return [ 'text', 'type', 'animation', 'typing', '打字', '文字' ];
	}

	public function get_style_depends(): array {
		return [ 'tooto-elementor' ];
	}

	public function get_script_depends(): array {
		return [ 'tooto-elementor' ];
	}

	protected function register_controls(): void {
		// Content Section
		$this->start_controls_section(
			'section_content',
			[
				'label' => __( '打字内容', 'tooto-elementor' ),
			]
		);

		$repeater = new \Elementor\Repeater();

		$repeater->add_control(
			'text_content',
			[
				'label' => __( '文本', 'tooto-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => __( 'Text typing effect', 'tooto-elementor' ),
				'label_block' => true,
			]
		);

		$repeater->add_control(
			'text_color',
			[
				'label' => __( '文本颜色', 'tooto-elementor' ),
				'type' => Controls_Manager::COLOR,
				'description' => __( '留空则使用默认颜色', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'typing_list',
			[
				'label' => __( '打字列表', 'tooto-elementor' ),
				'type' => Controls_Manager::REPEATER,
				'fields' => $repeater->get_controls(),
				'default' => [
					[
						'text_content' => __( 'Text typing effect', 'tooto-elementor' ),
					],
					[
						'text_content' => __( 'for your websites', 'tooto-elementor' ),
					],
					[
						'text_content' => __( 'Happy coding!', 'tooto-elementor' ),
					],
				],
				'title_field' => '{{{ text_content }}}',
			]
		);

		$this->add_control(
			'loop',
			[
				'label' => __( '循环播放', 'tooto-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'yes',
			]
		);

		$this->add_control(
			'start_on_visible',
			[
				'label' => __( '进入视口时开始', 'tooto-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
			]
		);

		$this->end_controls_section();

		// Settings Section
		$this->start_controls_section(
			'section_settings',
			[
				'label' => __( '打字设置', 'tooto-elementor' ),
			]
		);

		$this->add_control(
			'typing_speed',
			[
				'label' => __( '打字速度 (ms)', 'tooto-elementor' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 75,
				'min' => 10,
				'max' => 500,
			]
		);

		$this->add_control(
			'deleting_speed',
			[
				'label' => __( '删除速度 (ms)', 'tooto-elementor' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 30,
				'min' => 10,
				'max' => 500,
			]
		);

		$this->add_control(
			'start_delay',
			[
				'label' => __( '初始延迟 (ms)', 'tooto-elementor' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 0,
				'min' => 0,
			]
		);

		$this->add_control(
			'pause_duration',
			[
				'label' => __( '停顿时间 (ms)', 'tooto-elementor' ),
				'description' => __( '打完一行字后的停顿时间', 'tooto-elementor' ),
				'type' => Controls_Manager::NUMBER,
				'default' => 1500,
				'min' => 0,
			]
		);

		$this->add_control(
			'show_cursor',
			[
				'label' => __( '显示光标', 'tooto-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
			]
		);

		$this->add_control(
			'cursor_char',
			[
				'label' => __( '光标字符', 'tooto-elementor' ),
				'type' => Controls_Manager::TEXT,
				'default' => '|',
				'condition' => [
					'show_cursor' => 'yes',
				],
			]
		);
		
		$this->add_control(
			'hide_cursor_typing',
			[
				'label' => __( '打字时隐藏光标', 'tooto-elementor' ),
				'type' => Controls_Manager::SWITCHER,
				'default' => 'no',
				'condition' => [
					'show_cursor' => 'yes',
				],
			]
		);

		$this->end_controls_section();

		// Style Section
		$this->start_controls_section(
			'section_style_text',
			[
				'label' => __( '文本样式', 'tooto-elementor' ),
				'tab' => Controls_Manager::TAB_STYLE,
			]
		);

		$this->add_group_control(
			Group_Control_Typography::get_type(),
			[
				'name' => 'typography',
				'selector' => '{{WRAPPER}} .tooto-text-type__text, {{WRAPPER}} .tooto-text-type__cursor',
			]
		);

		$this->add_control(
			'default_text_color',
			[
				'label' => __( '默认文本颜色', 'tooto-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-text-type__text' => 'color: {{VALUE}};',
				],
			]
		);
		
		$this->add_group_control(
			Group_Control_Text_Shadow::get_type(),
			[
				'name' => 'text_shadow',
				'selector' => '{{WRAPPER}} .tooto-text-type__text',
			]
		);

		$this->add_control(
			'cursor_color',
			[
				'label' => __( '光标颜色', 'tooto-elementor' ),
				'type' => Controls_Manager::COLOR,
				'selectors' => [
					'{{WRAPPER}} .tooto-text-type__cursor' => 'color: {{VALUE}};',
				],
				'condition' => [
					'show_cursor' => 'yes',
				],
			]
		);

		$this->add_responsive_control(
			'align',
			[
				'label' => __( '对齐方式', 'tooto-elementor' ),
				'type' => Controls_Manager::CHOOSE,
				'options' => [
					'left' => [
						'title' => __( 'Left', 'tooto-elementor' ),
						'icon' => 'eicon-text-align-left',
					],
					'center' => [
						'title' => __( 'Center', 'tooto-elementor' ),
						'icon' => 'eicon-text-align-center',
					],
					'right' => [
						'title' => __( 'Right', 'tooto-elementor' ),
						'icon' => 'eicon-text-align-right',
					],
				],
				'selectors' => [
					'{{WRAPPER}} .tooto-text-type' => 'text-align: {{VALUE}};',
				],
			]
		);

		$this->end_controls_section();
	}

	protected function render(): void {
		$settings = $this->get_settings_for_display();

		$typing_list = [];
		if ( ! empty( $settings['typing_list'] ) ) {
			foreach ( $settings['typing_list'] as $item ) {
				$typing_list[] = [
					'text' => $item['text_content'],
					'color' => $item['text_color'],
				];
			}
		}

		if ( empty( $typing_list ) ) {
			return;
		}

		$config = [
			'strings' => $typing_list,
			'typeSpeed' => $settings['typing_speed'] ?? 75,
			'deleteSpeed' => $settings['deleting_speed'] ?? 30,
			'startDelay' => $settings['start_delay'] ?? 0,
			'pauseDuration' => $settings['pause_duration'] ?? 1500,
			'loop' => 'yes' === $settings['loop'],
			'startOnVisible' => 'yes' === $settings['start_on_visible'],
			'showCursor' => 'yes' === $settings['show_cursor'],
			'hideCursorWhileTyping' => 'yes' === $settings['hide_cursor_typing'],
		];

		$this->add_render_attribute( 'wrapper', 'class', 'tooto-text-type' );
		$this->add_render_attribute( 'wrapper', 'data-tooto-text-type-settings', wp_json_encode( $config ) );

		?>
		<div <?php $this->print_render_attribute_string( 'wrapper' ); ?>>
			<span class="tooto-text-type__text"></span>
			<?php if ( 'yes' === $settings['show_cursor'] ) : ?>
				<span class="tooto-text-type__cursor"><?php echo esc_html( $settings['cursor_char'] ); ?></span>
			<?php endif; ?>
		</div>
		<?php
	}
}
